//

// (C) Copyright 1999 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//

//
// DESCRIPTION:
// Dockable pane class implementation
//
// PURPOSE:
// This class, derived from CDialogBar is responsible for creating
// a dockable pane with an extra splitter to allow the user to size
// the pane when it is docked.
//
#include "stdafx.h"

#include "SampleListView.h"
#include "ToolBarEx.h"

#include "DockPane.h"
#include "resource.h"
#include "rxmfcapi.h"         // ACAD MFC stuff

#include "input.h"

/****************************************************************************/
/* MAP API INCLUDES */
/****************************************************************************/
#include <MapArxApi.h>
#include <MapSession.h>
#include <MapProj.h>
#include <MapODDefinition.h>
#include <MapODColumn.h>
#include <MapValue.h>
#include <MapODTable.h>
#include <MapODRecord.h>
#include <MapODIterator.h>
#include <MapValue.h>
#include <MapTemplate.h>
#include <gemat3d.h>
#include <MapStringArray.h>
#include <MapConstants.h>
#include <accmd.h>
#include <adscodes.h>
#include <rxregsvc.h>  
/****************************************************************************/

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// A custom message to work around a problem getting the miniframe pointer
#define WM_DOCKSTATE_CHANGED (WM_USER + 17)
#define MYCOMBO_WIDTH 120 //the width of the combo boxes

// Minimum size (in pixels) of the dockable pane when floating
static const int kMinFloatingHeight = 10;
static const int kMinFloatingWidth = 60;

//////////////////////////////////////////////////////////////////////////////
// CDockPaneWindowHook
//
// The purpose of this class is to hook the floating mini frame so we can
// intercept the AutoCAD custom message WM_ACAD_KEEPFOCUS
IMPLEMENT_DYNAMIC(CDockPaneWindowHook, CSubclassWnd);

LRESULT CDockPaneWindowHook::WindowProc(UINT msg, WPARAM wp, LPARAM lp)
{
	switch(msg) {
	    case WM_ACAD_KEEPFOCUS:
            return TRUE;
        default:break;
    }
	return CSubclassWnd::WindowProc(msg, wp, lp);
}

//////////////////////////////////////////////////////////////////////////////
// CDockPane

IMPLEMENT_DYNAMIC(CDockPane, CAdUiDockControlBar)

CDockPane::CDockPane()
{
	m_pListView = NULL;
	m_bViewOD = FALSE;
	m_bUseToolTips = FALSE;
}

CDockPane::~CDockPane()
{
	if(NULL != m_pListView)
		delete m_pListView;
	
	if (m_MiniFrameHook.IsHooked())
	{
		// Unhook the miniframe window
		m_MiniFrameHook.HookWindow(NULL);
	}

}

BOOL CDockPane::Create(CWnd *pParentWnd)
{
	// For now, the toolID will be hard-coded, until Alain gives us a unique value
	// AutoCAD uses this ID to retrieve the location of the pane in the registry
	UINT nControlBarID = AFX_IDW_CONTROLBAR_FIRST + 5;

	// Create the dockable pane window
	CString sWndClass;
	sWndClass = AfxRegisterWndClass(CS_DBLCLKS);
	CRect rect(0, 0, 250, 200);
	if (!CAdUiDockControlBar::Create(sWndClass,
		NULL,
		WS_VISIBLE|WS_CHILD|WS_CLIPCHILDREN,
		rect,
		pParentWnd, nControlBarID))
	{
		return FALSE;
	}

	// Set the window title
	CString sTitle;
	VERIFY(sTitle.LoadString(IDS_DOCKTITLE));
	SetWindowText(sTitle);
	
	// Move the dockable pane window to the last known location
	EnableDocking(CBRS_ALIGN_LEFT|CBRS_ALIGN_RIGHT);
	RestoreControlBar();

	return TRUE;
}

int CDockPane::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
	if (CAdUiDockControlBar::OnCreate(lpCreateStruct) == -1)
	{
		return -1;
	}
	const char *pszContentClass =
		AfxRegisterWndClass(CS_HREDRAW|CS_VREDRAW,
		LoadCursor(NULL, IDC_ARROW));

	m_pListView = new CSampleListView();

	DWORD dwStyle = WS_CHILD | WS_BORDER | WS_DLGFRAME;
	CRect rect(0, 0, 0, 0);
		
	//create the list view;
	if(!m_pListView->Create(NULL, NULL, dwStyle, rect, this, 777, NULL))
	{
		TRACE0("Failed to create m_pListView\n");
		acutPrintf( "\nFailed to create m_pListView\n." );
		delete m_pListView;
		m_pListView = NULL;
		return -1;		// fail to create
	}

	//create the toolbar
	if (!m_wndToolBar.Create(this) ||
		!m_wndToolBar.LoadToolBar(IDR_MAINFRAME))
	{
		TRACE0("Failed to create toolbar\n");
		return -1;      // fail to create
	}

	// TODO: Remove this if you don't want tool tips or a resizeable toolbar
	m_wndToolBar.SetBarStyle(m_wndToolBar.GetBarStyle() |
		CBRS_TOOLTIPS | CBRS_FLYBY );

	/////////////////////////////////////////
	//Create the Combobox for Tracking On/Off
	/////////////////////////////////////////
	
	//First get the index of the placeholder's position in the toolbar
    int index = 0;
    while(m_wndToolBar.GetItemID(index)!=ID_VIEW_OBJECTDATA) index++;
    
	//next convert that button to a seperator and get its position
    m_wndToolBar.SetButtonInfo(index, ID_VIEW_OBJECTDATA, TBBS_SEPARATOR, MYCOMBO_WIDTH);
    m_wndToolBar.GetItemRect(index, &rect);
    
	//expand the rectangle to allow the combo box room to drop down
    rect.top+=2;
    rect.bottom += 100;
    // then .Create the combo box and show it
    if (!m_wndToolBar.m_wndOnOffCmb.Create(WS_CHILD|WS_VISIBLE | CBS_AUTOHSCROLL | 
                                       CBS_DROPDOWNLIST | CBS_HASSTRINGS ,
                                       rect, &m_wndToolBar, IDC_ONOFF_COMBO))
    {
		TRACE0("Failed to create On/Off combo-box\n");
		return FALSE;
    }

	////////////////////////////////////
	//Create the Combobox for OD Tables
	////////////////////////////////////

	//First get the index of the placeholder's position in the toolbar
    while(m_wndToolBar.GetItemID(index)!=ID_VIEW_ODTABLES) index++;
    
	//next convert that button to a seperator and get its position
    m_wndToolBar.SetButtonInfo(index, ID_VIEW_OBJECTDATA, TBBS_SEPARATOR, MYCOMBO_WIDTH);
    m_wndToolBar.GetItemRect(index, &rect);
    
	//expand the rectangle to allow the combo box room to drop down
    rect.top+=2;
    rect.bottom += 100;
    // then .Create the combo box and show it
    if (!m_wndToolBar.m_wndODTableCmb.Create(WS_CHILD|WS_VISIBLE | CBS_AUTOHSCROLL | 
                                       CBS_DROPDOWNLIST | CBS_HASSTRINGS ,
                                       rect, &m_wndToolBar, IDC_ODTABLE_COMBO))
    {
		TRACE0("Failed to create On/Off combo-box\n");
		return FALSE;
    }

	m_wndToolBar.m_wndOnOffCmb.ShowWindow(SW_SHOW);
	m_wndToolBar.m_wndODTableCmb.ShowWindow(SW_SHOW);
    
	//fill On/Off the combo box
    m_wndToolBar.m_wndOnOffCmb.AddString("Tracking OFF");
    m_wndToolBar.m_wndOnOffCmb.AddString("Tracking ON");
    m_wndToolBar.m_wndOnOffCmb.SetCurSel(0);

	//fillOD Tables the combo box
    m_wndToolBar.m_wndODTableCmb.AddString("None");
    m_wndToolBar.m_wndODTableCmb.SetCurSel(0);
	

	m_pListView->Init();
	m_pListView->ShowWindow(SW_SHOW);

	return 0;
}

void CDockPane::SizeChanged(CRect *lpRect, BOOL bFloating, int flags)
{
	if(!m_bFloating) //docked
	{
		
		// size the listview
		lpRect->InflateRect(-6, -6);
		m_pListView->SetWindowPos( NULL,
			lpRect->left, lpRect->top + 35,
			lpRect->Width(), lpRect->Height() - 35,
			SWP_NOZORDER);

		// size the toolbar
		m_wndToolBar.SetWindowPos( NULL, 1, 20,
			lpRect->Width(), lpRect->Height() - 40,
			SWP_NOZORDER);
	}
	else //floating
	{
		// size the listview
		lpRect->InflateRect(-6, -6);
		m_pListView->SetWindowPos( NULL,
			lpRect->left, lpRect->top + 45,
			lpRect->Width(), lpRect->Height() - 45,
			SWP_NOZORDER);

		// size the toolbar
		m_wndToolBar.SetWindowPos( NULL, 1, 10,
			lpRect->Width(), lpRect->Height() - 30,
			SWP_NOZORDER);
	}
}

BEGIN_MESSAGE_MAP(CDockPane, CAdUiDockControlBar)
    //{{AFX_MSG_MAP(CDockPane)
    ON_WM_CREATE()
	ON_CBN_SELENDOK(IDC_ONOFF_COMBO, OnSelEndOk_OnOff)
	ON_CBN_SELCHANGE(IDC_ONOFF_COMBO, OnSelChange_OnOff)
	ON_CBN_SELENDOK(IDC_ODTABLE_COMBO, OnSelEndOk_ODT)
	ON_CBN_SELCHANGE(IDC_ODTABLE_COMBO, OnSelChange_ODT)
	//}}AFX_MSG_MAP
	ON_MESSAGE(WM_DOCKSTATE_CHANGED, OnChangedDockedState)
END_MESSAGE_MAP()


void CDockPane::ToggleBarState(BOOL bShrinkIfDocked /* = TRUE */)
{
}

BOOL CDockPane::IsWindowVisible() const
{
    if (!CAdUiDockControlBar::IsWindowVisible())
        return FALSE;

    return TRUE;
}

void CDockPane::HookMiniFrame()
{
    CWnd* pMiniFrameWnd = GetParentFrame();
    m_MiniFrameHook.HookWindow(pMiniFrameWnd);
}

LONG CDockPane::OnChangedDockedState(UINT, LONG)
{
    //if we are not floating, unhook the window procedure, otherwise hook it
    if(m_bFloating)
        HookMiniFrame();
    return 0L;
}

void CDockPane::OnBarStyleChange(DWORD oldStyle, DWORD newStyle)
{
    m_bFloating = (newStyle & CBRS_FLOATING);
    //This is lame - when this notification comes we can't get a pointer
    //to the miniframe even though its already been created, so we'll
    //post a message to ourselves to swtich it.
    PostMessage(WM_DOCKSTATE_CHANGED);
}

////////////////////////////////////////////////////////////////////////////
// This method iterates over all the Object Data tables in the drawing
// and updates the drop down list boxes in the docking window.
////////////////////////////////////////////////////////////////////////////
Adesk::Boolean CDockPane::OnLoadDwg()
{
	//Get all the OD tables for this drawing
	AcMap::EErrCode ecode;

	ClearList(); //clear out the listview

	AcMapSession *pMapSession = AcMapGetSession();
	if (pMapSession != NULL) 
	{
		AcMapODContainer *pODContainer = NULL;
		AcMapProject *pProj = NULL;
		if (pMapSession->GetProject(pProj) == Adesk::kFalse)
		{
		   pMapSession = NULL;
		   TRACE0(" Can't get MAP Project.");
		   throw 1;
		}

		if (pProj->GetODContainer(pODContainer) == Adesk::kFalse)
		{
		   TRACE0(" Can't get MAP OD interface.");
		   throw 1;
		}

		int iTableCount = pODContainer->CountODTables();
		if(iTableCount > 0)
		{
			AcMapStringArray tableNames;
			AcMapODTable *pODTable = NULL;
			ecode =  pODContainer->GetODTableNames(tableNames);
			if(ecode == AcMap::kOk)
			{
				m_wndToolBar.m_wndODTableCmb.ResetContent(); //clear the list
				
				for(int i=0; i < iTableCount; i++)
				{
					//clear the list
					ClearList();
					
					//fill the combo box with the names
					m_wndToolBar.m_wndODTableCmb.AddString(tableNames[i]);

					//update the list to show the field names
					ecode = pODContainer->GetODTable(pODTable, tableNames[i]);
					if(ecode == AcMap::kOk)
					{
						GetTableFieldNames(pODTable);
					}
					if(pODTable)
					{
						delete pODTable;
						pODTable = NULL;
					}
				}
			}
			else //set to "None"
			{
				//TODO
			}

			int iItem = m_wndToolBar.m_wndODTableCmb.GetCount();
			m_wndToolBar.m_wndODTableCmb.SetCurSel(iItem-1);
		}
	}

	return Adesk::kTrue;

}

Adesk::Boolean CDockPane::OnUnloadDwg()
{
	return Adesk::kTrue;
}

void CDockPane::SetFloatingSize(const CSize& size)
{
	// TODO
}

void CDockPane::OnSaveComplete(AcDbDatabase *pDwg, const char *pActualName)
{
	// TODO
}

void CDockPane::PostReactorMessage(WPARAM wParam, LPARAM lParam)
{
	//TODO
}

/////////////////////////////////////////////////////////////////
// This is the main work horse for getting object data from an
// object.
/////////////////////////////////////////////////////////////////
void CDockPane::GetObjectData(const AcDbObjectId pObjId)
{
	m_strToolTip.Empty();
	
	AcMapSession *pMapSession = AcMapGetSession();
	if (pMapSession != NULL) 
	{
		AcMapODContainer *pODContainer = NULL;
		AcMapProject *pProj = NULL;
		if (pMapSession->GetProject(pProj) == Adesk::kFalse)
		{
		   pMapSession = NULL;
		   TRACE0(" Can't get MAP Project.");
		   throw 1;
		}

		if (pProj->GetODContainer(pODContainer) == Adesk::kFalse)
		{
		   TRACE0(" Can't get MAP OD interface.");
		   throw 1;
		}

		AcMapODRecordIterator *pIter = NULL;
   
		try
		{
			// Get iterator
			//
			if (pODContainer->GetObjectODRecordIterator (pIter) != AcMap::kOk)
			{
			   acutPrintf ("\n Can't get object iterator.\n");
			   throw 1;
			}

			AcDbObjectId id = pObjId;
		   
			// Initialize iterator
			//
			if (pIter->Init (id, AcMap::kOpenForRead, Adesk::kFalse) != AcMap::kOk)
			{
			   acutPrintf ("\n Can't initialize iterator by ACAD object ID.\n");
			   throw 1;
			}
   
			// Check open mode
			//
			if (pIter->OpenMode() != AcMap::kOpenForRead)
			{
			   acutPrintf ("\n Wrong iterator mode.\n");
			   throw 1;
			}
   
			// Iterate through all records
			//
			AcMapODTableRecord record;
			for (; pIter->IsDone() == Adesk::kFalse; pIter->Next())
			{
				// Get record
				//
				if (pIter->GetRecord (record) != AcMap::kOk)
				{
				   acutPrintf ("\n Can't get record.\n");
				   throw 1;
				}

				// Get the table
				//
				AcMapODTable *pTable = NULL;
				if (pODContainer->GetODTable (pTable, record.ODTableName()) != AcMap::kOk)
				{
					acutPrintf ("\n Can't get the table.\n");
					throw 1;
				}

				//if this table is not selected as a filter for viewing then return
				if(!IsTableSelected(pTable->Name()))
				{
					delete pTable;
					pTable = NULL;

					delete pIter;
					pIter = NULL;
					
					return;
				}
				
				int iVal=0;
				double dVal;
				
				CString str;
				
				// Get record info
				for (int i = 0; i < record.Count(); i++)
				{
					AcMapODTableDefinition tableDef = pTable->Definition();
					AcMapODColumnDefinition Column;

					if(tableDef.GetColumn(Column, i) != AcMap::kOk)
						break;

					const char* colName = Column.Name();
					
					AcMapValue &val = record.Value (i);
					switch (val.Type ())
					{
					case AcMap::kInteger:
						iVal = (int)val;
						str.Format("%d", iVal);
						m_pListView->AddItem(i,1, str);
					break;

					case AcMap::kReal:
						dVal = (double)val;
						str.Format("%f", dVal);
						m_pListView->AddItem(i,1, str);
					break;

					case AcMap::kCharacter:
					{
						str = ((const char *)val);
						m_pListView->AddItem(i,1, str);
					}
					break;
       
					case AcMap::kPoint:
					{
						const AcGePoint3d pnt3 = (const AcGePoint3d &)val;
						double dx = ((const AcGePoint3d &)val).x;
						double dy = ((const AcGePoint3d &)val).y;
						double dz = ((const AcGePoint3d &)val).z;
						str.Format("%f, %f, %f", dx, dy, dz);
						m_pListView->AddItem(i,1, str);
					}
					break;

					default:
						acutPrintf ("\n Wrong data type\n");
						throw 1;
					}

					//if it's check, then add to the tooltip
					if(m_pListView->IsItemChecked(i))
					{
						if(m_strToolTip.GetLength() != 0)
						{
							m_strToolTip += ", ";
							m_strToolTip += str;
						}
						else
						{
							m_strToolTip += str;
						}
					}
				}

				// Delete table object
				//
				delete pTable;
				pTable = NULL;
			}
   
			// Delete iterator
			//
			delete pIter;
			pIter = NULL;
	
		}
		catch (...)
		{
			if(pIter)
				delete pIter;
		}
	}
}

/////////////////////////////////////////////////////////////////
// Get the field names for the specified table
/////////////////////////////////////////////////////////////////
void CDockPane::GetTableFieldNames(AcMapODTable *pODTable)
{
	ASSERT(NULL != pODTable);
	if(NULL == pODTable)
		return;

	AcMap::EErrCode ecode;
	AcMapODTableDefinition tableDef;
	tableDef = pODTable->Definition();
	
	int iColCount = tableDef.Count();
	if(iColCount)
	{
		for(int i=0; i < iColCount; i++)
		{
			AcMapODColumnDefinition Column;
			ecode = tableDef.GetColumn(Column, i);
			if(ecode == AcMap::kOk)
			{
				//populate the list with the field names
				const char* pName = Column.Name();
				m_pListView->AddItem(i,0, pName);
			}
		}
	}
}

/////////////////////////////////////////////////////////////////
// Clear the list view
/////////////////////////////////////////////////////////////////
void CDockPane::ClearList()
{
	ASSERT(m_pListView);
	
	if(m_pListView)
		m_pListView->ClearList();
}

/////////////////////////////////////////////////////////////////
// Post process the selection and save the index
/////////////////////////////////////////////////////////////////
void CDockPane::OnSelEndOk_OnOff()
{
	m_cbOnOffIndex = m_wndToolBar.m_wndOnOffCmb.GetCurSel();
}

/////////////////////////////////////////////////////////////////
// User has turned on/off the selection
/////////////////////////////////////////////////////////////////
void CDockPane::OnSelChange_OnOff()
{	
	int nCase = m_cbOnOffIndex;
		
	switch(nCase)
	{
		case 0:
			EndWatch();
			break;
		
		case 1:
			StartWatch();
			break;
	
		default:
			break;
	}
}

/////////////////////////////////////////////////////////////////
// Post process the selection and save the index
/////////////////////////////////////////////////////////////////
void CDockPane::OnSelEndOk_ODT()
{
	m_cbODTableIndex = m_wndToolBar.m_wndODTableCmb.GetCurSel();
}

/////////////////////////////////////////////////////////////////
// User has selected another table from the list
/////////////////////////////////////////////////////////////////
void CDockPane::OnSelChange_ODT()
{	
	AcMap::EErrCode ecode;
	CString tableName;
	
	ClearList();
	
	m_wndToolBar.m_wndODTableCmb.GetWindowText(tableName);

	AcMapSession *pMapSession = AcMapGetSession();
	if (pMapSession != NULL) 
	{
		AcMapODContainer *pODContainer = NULL;
		AcMapProject *pProj = NULL;
		if (pMapSession->GetProject(pProj) == Adesk::kFalse)
		{
		   pMapSession = NULL;
		   TRACE0(" Can't get MAP Project.");
		   throw 1;
		}

		if (pProj->GetODContainer(pODContainer) == Adesk::kFalse)
		{
		   TRACE0(" Can't get MAP OD interface.");
		   throw 1;
		}

		AcMapODTable *pODTable = NULL;
		ecode = pODContainer->GetODTable(pODTable, tableName);
		if(ecode == AcMap::kOk)
		{
			GetTableFieldNames(pODTable);
		}
		if(pODTable)
		{
			delete pODTable;
			pODTable = NULL;
		}
	}
}

/////////////////////////////////////////////////////////////////
// Starting the watch, so add the point monitor and turn on force
// picking.
/////////////////////////////////////////////////////////////////
void CDockPane::StartWatch() 
{
	Acad::ErrorStatus es;

	m_ipm.m_oldId.setNull();
	m_ipm.SetDlgWnd(this);
	
	es = curDoc()->inputPointManager()->addPointMonitor(&m_ipm);
	if(es != Acad::eOk)
	{
		acutPrintf("\nCant't add point monitor.\n");
		return;
	}

	curDoc()->inputPointManager()->turnOnForcedPick();
}

/////////////////////////////////////////////////////////////////
// remove the point monitor
/////////////////////////////////////////////////////////////////
void CDockPane::EndWatch() 
{
	curDoc()->inputPointManager()->turnOffForcedPick();
	curDoc()->inputPointManager()->removePointMonitor(&m_ipm);
	curDoc()->inputPointManager()->revokePointFilter();
}

/////////////////////////////////////////////////////////////////
// Turn tool tips on
/////////////////////////////////////////////////////////////////
BOOL CDockPane::UseToolTips()
{
	return m_pListView->UseToolTips();
}

/////////////////////////////////////////////////////////////////
// Get the tool tip string
/////////////////////////////////////////////////////////////////
BOOL CDockPane::GetToolTipString(CString &str)
{
	BOOL bRes = TRUE;
	
	str = m_strToolTip;

	if(str.IsEmpty())
		bRes = FALSE;
	
	return bRes;
}

/////////////////////////////////////////////////////////////////
// Simple check to see if the table is selected
/////////////////////////////////////////////////////////////////
BOOL CDockPane::IsTableSelected(const CString str)
{
	BOOL bRes = FALSE;
	CString tableName;
	
	m_wndToolBar.m_wndODTableCmb.GetWindowText(tableName);

	if(str == tableName)
		bRes = TRUE;
	
	return bRes;
}


